############# Quantitative Textanalyse mit dem Paket quanteda ##############
############# KeyWord-in-Context-Analysen (KWIC) und n-Gramme ##############

# Zunächst und für alle Fälle: Arbeitsbereich komplett bereinigen (Housekeeping)
rm(list= ls(all= TRUE))

# Notwendige Pakete installieren
# Falls Sie hier irgendwo gefragt werden:
# Do you want to install from sources the packages which need compilation?
# mit "Nein" antworten: y/n: n
install.packages("tm")
install.packages("quanteda") 
install.packages("quanteda.textplots")
install.packages("quanteda.textstats")
install.packages("quanteda.textmodels")
install.packages("readtext")
install.packages("tidytext")
install.packages("dplyr")
install.packages("ggplot2")
install.packages("ggthemes")
install.packages("tidyr")
install.packages("widyr")
install.packages("igraph")
install.packages("ggraph")
install.packages("wordcloud")
install.packages("RColorBrewer")
install.packages("proxy")

# Aufrufen einiger dieser Pakete
library(quanteda)
library(quanteda.textplots)
library(quanteda.textstats)
library(quanteda.textmodels)
library(readtext)

# Das Arbeitsverzeichnis auswählen; in RStudio 
# Session > Set Working Directory > Choose Directory 
# oder den Dateipfad manuell eintragen: setwd("C:/Users/.../ToxizitaetsWorkshop")

# Quanteda arbeitet mit drei basalen Typen von Objekten:
# Korpus
#   Speichert Zeichenketten und Variablen in einem data frame
#   Kombiniert Texte mit Variablen auf Dokumentebene
# Tokens
#   Speichert Token in einer Liste von Vektoren
#   Effizienter als Zeichenketten, aber die Positionen der Wörter 
#   bleiben erhalten
#   Die Positionsanalyse (in Wortfolgen) wird mit textstat_collocations(), 
#   tokens_ngrams() und tokens_select() oder fcm() 
#   mit der Option window durchgeführt
# Document-feature matrix (DFM)
#   Stellt die Häufigkeit von Merkmalen in Dokumenten in einer Matrix dar
#   Stellt die effizienteste Struktur dar, aber sie enthält 
#   keine Informationen über die Position der Wörter
#   Nicht-positionale (bag-of-words) Analysen werden mit vielen 
#   der textstat_* und textmodel_* Funktionen durchgeführt

# Der workflow in quanteda läuft immer über das Einlesen der Textdateien
# über die Korpuserstellung zur Tokenisierung, dann ggf. weiter zur 
# Erstellung eines dfm

# Die Texte in einem Unterordner mit readtext einlesen (txt-Format, UTF-8 codiert)
myCorpus <- readtext("CorpusKriegssammlung/*", encoding = "UTF-8")

# Das Korpus mit quanteda erstellen
ToxiCorpus <- corpus(myCorpus) 
class(ToxiCorpus)
# "corpus"    "character"

summary(ToxiCorpus, n = 5) # Metadaten / Überblick über die ersten 5 Dokumente im Korpus: 
# Anzahlen Types, Tokens, Sätze je Dokument

# Hier können zusätzlich noch Metadaten zu den Dateinamen eingegeben werden
docvars(ToxiCorpus, 'published') <- c("1904", "1898", "1907", "1894", "1907", 
                                      "1915", "1915", "1916", "1867", "1917", 
                                      "1916", "1919", "1914", "1918", "1917", 
                                      "1901", "1916", "1916", "1917", "1918", 
                                      "1915", "1917", "1918", "1915", "1914", 
                                      "1917", "1887", "1917", "1885", "1910") 

docvars(ToxiCorpus, 'category') <- c("Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918, Kinder- und Jugendbücher", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke", 
                                     "Historische Drucke", 
                                     "Historische Drucke, Geschichte / Ethnographie / Geographie") 

docvars(ToxiCorpus, 'publisher') <- c("Sattler", "Walther", "Lehmann", "Hölder", 
                                      "Stalling", "Stilke", "Verlag der Weißen Bücher", 
                                      "Reimer", "Verlag des Missionshauses", "Mittler", 
                                      "Nister", "Hobbing", "Vieweg", "Stuttgart", 
                                      "Stilke", "Jansen", "Reichenbach", "Reiche", 
                                      "Frankenstein & Wagner", "Reichsdruckerei", 
                                      "Duncker", "Hofer", "Steinkopf", "Mutze", 
                                      "Weichert", "Süsserott", "Walther & Apolant", 
                                      "Druck und Verlag der Missionshandlung", 
                                      "Geographisches Institut", "Levysohn") 

docvars(ToxiCorpus, 'comment') <- c("P_Praeventiv", "P_Praeventiv", "P_Praeventiv", 
                                    "P_Praeventiv", "P_Praeventiv", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeAllg_bis1920", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeAllg_bis1920", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeAllg_bis1920", 
                                    "PDruckeAllg_bis1920", "PDruckeAllg_bis1920", 
                                    "PDruckeAllg_bis1920") 


head(docvars(ToxiCorpus)) # den Kopf der Metadaten betrachten

# Will man einzelne Metadaten betrachten, kann man das Feld direkt benennen
docvars(ToxiCorpus, field = "published")

# Alternative Formulierung
ToxiCorpus$published

# Variablen auf Dokumentebene ermöglicht es beispielsweise, ein subset herzustellen, 
# indem Dokumente nach dem Publikationsjahr ausgewählt werden:
ToxiCorpus_recent <- corpus_subset(ToxiCorpus, published >= 1915)
ndoc(ToxiCorpus_recent)
# jetzt sind nur noch 18 Dokumente in myCorpus_recent enthalten; welche?
docnames(ToxiCorpus_recent)

# Oder wir suchen uns Texte nach Sammlung aus
ToxiCorpus_recent <- corpus_subset(ToxiCorpus, comment == "PDruckeEuropeana1914-1918")
docnames(ToxiCorpus_recent)

# Die Einheit eines Corpus verändern
# corpus_reshape() ermöglicht es, die Einheiten von Texten 
# zwischen Dokumenten, Absätzen und Sätzen zu ändern. 
# Da es Dokument-IDs aufzeichnet, können Texte 
# in der ursprünglichen Einheit wiederhergestellt werden, 
# auch wenn das Korpus durch andere Funktionen verändert wird.

# Die Grundlage der Einheiten nach Sätze verändern
ToxiCorpus_sentence <- corpus_reshape(ToxiCorpus, to = "sentences")
print(ToxiCorpus_sentence)[1:6]
# andere Möglichkeiten: documents, paragraphs


#### Tokens 
# Ein Tokens Objekt erstellen
# tokens() segmentiert Texte in einem Korpus nach Wortgrenzen
# in Token (Wörter oder Sätze).
# Standardmäßig segmentiert tokens() entlang von Trennzeichen 
# (in der Regel Leerzeichen)
ToxiCorpus_token <- tokens(ToxiCorpus)

# Zahlen und Satzzeichen entfernen
ToxiCorpus_token <- tokens(ToxiCorpus_token, remove_numbers = TRUE, remove_punct = TRUE)

# nun setzen wir noch alles in Kleinschreibung (lowercasing)
ToxiCorpus_token <- tokens_tolower(ToxiCorpus_token)

###################### KeyWord-in-Context-Analysen ##########################
# Hier kann man in einer mit kwic() erstellten Konkordanzansicht sehen, 
# wie Schlüsselwörter in den tatsächlichen Kontexten verwendet werden.

# da wir die Texte bereits in Kleinschreibung gesetzt haben, müssen wir 
# das Schlüsselwort auch kleinschreiben

keyword_neger <- kwic(ToxiCorpus_token, pattern =  "neger")
head(keyword_neger, 10) # die ersten 10 KWICs anschauen

# Das geht hübscher mit View()
View(keyword_neger)

# Das Fenster um die Schlüsselworte herum lässt sich mit "window" verändern
# (voreingestellt sind 5 Worte):

keyword_neger <- kwic(ToxiCorpus_token, pattern =  "neger", window = 7)
View(keyword_neger)

# es können auch wildcards (*) verwendet werden

keyword_negerWildcard <- kwic(ToxiCorpus_token, pattern =  "neger*", window = 7)

# KWIC-Suchen können auch mit mehreren Worten durchgeführt werden. 
# Will man diese finden, dann müssen
# die Worte durch ein Leerzeichen getrennt und von phrase() umschlossen werden.
keyword_dieNeger <- kwic(ToxiCorpus_token, pattern = phrase("die neger*"))
keyword_dieNeger

keyword_multiword <- kwic(ToxiCorpus_token, pattern = phrase(c("die neger*", "die kaffer*")))



############################### N-Gramme ##################################
# Mit tokens_ngrams() können wir aus Token n-Gramme 
# in beliebiger Länge erzeugen. N-Gramme sind eine Folge von Token 
# aus bereits tokenisierten Textobjekten.
# Hier schauen wir uns Bigramme an (n = 2)
ToxiCorpus_token_bigram <- tokens_ngrams(ToxiCorpus_token, n = 2)

# Die ersten 30 Treffer anschauen
head(ToxiCorpus_token_bigram[[1]], 30)

# Das ist erst einmal wenig interessant. 
# Während tokens_ngrams() n-Gramme in allen möglichen Kombinationen von Token erzeugt, 
# erzeugt tokens_select() selektive n-Gramme. So können Sie beispielsweise 
# mit phrase() und einem Platzhalter (*) Bigramme nach bestimmten Begriffen filtern.
ToxiCorpus_token_neger_select <- tokens_select(ToxiCorpus_token_bigram, pattern = phrase("neger_*"))
head(ToxiCorpus_token_neger_select[[1]], 30)

# um die häufigsten n-Gramme anzuschauen, benutzen wir die Funktion 
# textstat_frequencys() aus dem Paket quanteda.textstats
bigram_dfm <- dfm(ToxiCorpus_token_bigram)
bigram_freq <- textstat_frequency(bigram_dfm)

# Wordcloud, um die häufigsten Bigramme zu visualisieren
textplot_wordcloud(bigram_dfm,max_words = 100,
                   ordered_color = TRUE)

# Alle Bigramme anzeigen, in denen die Zeichenfolge "neger" mehr als zwei Mal vorkommt 
bigram_freq[(grepl("neger*", bigram_freq$feature)) & bigram_freq$frequency > 2,]


# Ganz analog können wir uns Trigramme anschauen:
ToxiCorpus_token_trigram <- tokens_ngrams(ToxiCorpus_token, n = 3)

ToxiCorpus_token_negertri_select <- tokens_select(ToxiCorpus_token_trigram, pattern = phrase("neger_*_*"))
head(ToxiCorpus_token_negertri_select[[1]], 30)

# die häufigsten Trigramme
trigram_dfm <- dfm(ToxiCorpus_token_trigram)
trigram_freq <- textstat_frequency(trigram_dfm)

# Wordcloud, um die häufigsten Trigramme zu visualisieren
textplot_wordcloud(trigram_dfm,max_words = 100,
                   ordered_color = TRUE)

# Alle Trigramme anzeigen, in denen die Zeichenfolge "neger" mehr als zwei Mal vorkommt 
trigram_freq[(grepl("neger*", trigram_freq$feature)) & trigram_freq$frequency > 2,]


##### Korpusanalyse mit einem Diktionär
# In quanteda kann man sich vergleichsweise einfach selbst ein Diktionär 
# mit verschiedenen Kategorien anlegen
dict <- dictionary(list(Rasse = c("bastard*", "blendl*", "halbblut", "mestiz*", 
                                  "mischrass*", "mulatte", "negr*"),
                        Sexualität = c("androgyn*", "dandy", "hermaphro*", 
                                       "homophil*", "homosex*", "masturb*", 
                                       "masochis*", "pädera*", "pervers*", "sadism*"),
                        Diskriminierung = c("affenmensch*", "aussätzig*", "barbar*", 
                                            "behind*", "kannib*", "invalid*", 
                                            "liliput*", "missgebild*", 
                                            "rückständ*", "wild*", "zwerg*"),
                        Ethnie = c("apache", "berber", "botokude", "eskimo", 
                                   "hottentott*", "indian*", "kaffer*", "kanak*", 
                                   "kreol*", "marron*", "cimarron*", "mohr", 
                                   "mongol*", "neger*", "pygmäe*", "zigeuner*")))
print(dict)

# Die im Diktionär enthaltenen Worte innerhalb des Textkorpus nachschauen lassen
ToxiCorpus_dict <- tokens_lookup(ToxiCorpus_token, dictionary = dict)

# Die Worte aus dem Diktionär, die im Korpus gefunden wurden, lassen sich 
# als kwic ausgeben
kwic_Sexualitaet <- kwic(ToxiCorpus_token, dict["Sexualität"])
View(kwic_Sexualitaet)

kwic_Diskriminierung <- kwic(ToxiCorpus_token, dict["Diskriminierung"])
View(kwic_Diskriminierung)

# Und die Häufigkeit der im Diktionär enthaltenen Wörter 
# pro Dokument kann ausgezählt werden
Uebersicht_dict <- convert(dfm(ToxiCorpus_dict), to = "data.frame")
View(Uebersicht_dict)


kwic_Rasse <- kwic(ToxiCorpus_token, dict["Rasse"])
View(kwic_Rasse)




######### Coda: Metadaten für die beiden weiteren Korpora #######################

# Für das Kiautschou-Korpus

# Hier können zusätzlich noch Metadaten zu den Dateinamen eingegeben werden
docvars(ToxiCorpus, 'published') <- c("1901", "1898", "1897", "1901", "1901", 
                                      "1899", "1901", "1902", "1901", "1898", 
                                      "1898", "1901", "1905", "1901", "1914", 
                                      "1917", "1916", "1916", "1915", "1915", 
                                      "1917", "1917", "1915", "1917", "1916", 
                                      "1914", "1919", "1915", "1907", "1901") 

docvars(ToxiCorpus, 'category') <- c("Ostasiatica", "Ostasiatica", "Ostasiatica", 
                                     "Ostasiatica", "Ostasiatica", "Ostasiatica", 
                                     "Ostasiatica", "Ostasiatica", "Ostasiatica", 
                                     "Ostasiatica", "Ostasiatica", "Ostasiatica", 
                                     "Ostasiatica", "Ostasiatica", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke, Krieg 1914-1918 ", 
                                     "Historische Drucke digital", 
                                     "Historische Drucke digital") 

docvars(ToxiCorpus, 'publisher') <- c("Reiff", "Reimer", "Stilke", "Berliner Stadtmission", 
                                      "Buchh. der Berliner evang. Missionsgesellschaft", 
                                      "Paulinus-Druck", "Strauß", "Köhler", 
                                      "Mittler & Sohn", "Weber", "Schall & Grund", 
                                      "Deutscher Verlag", "Lax", "Zieger", "Wegner", 
                                      "Weichert", "Bertelsmann", "Stalling", 
                                      "Mittler & Sohn", "Tageblatt für Nord-China", 
                                      "Kolonial-Wirtschaftliches Komitee", 
                                      "Curtius", "Kolonial-Wirtschaftliches-Komitee", 
                                      "Weicher", "Schöningh", "Meidinger", "ohne Verlag", 
                                      "Berliner Evang. Missionsgesellschaft", 
                                      "Carl Siwinna Phönix-Verlag", 
                                      "Deutscher Kolonial-Verlag") 

docvars(ToxiCorpus, 'comment') <- c("PDruckeSOA_WEST", "PDruckeSOA_WEST", "PDruckeSOA_WEST", 
                                    "PDruckeSOA_WEST", "PDruckeSOA_WEST", "PDruckeSOA_WEST", 
                                    "PDruckeSOA_WEST", "PDruckeSOA_WEST", "PDruckeSOA_WEST", 
                                    "PDruckeSOA_WEST", "PDruckeSOA_WEST", "PDruckeSOA_WEST", 
                                    "PDruckeSOA_WEST", "PDruckeSOA_WEST", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918",
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeEuropeana1914-1918", "PDruckeEuropeana1914-1918", 
                                    "PDruckeAllg_bis1920", "PDruckeAllg_bis1920") 


# Für das Kinderbücher-Korpus

# Hier können zusätzlich noch Metadaten zu den Dateinamen eingegeben werden
docvars(ToxiCorpus, 'published') <- c("1914", "1906", "1916", "1914", "1909", 
                                      "1917", "1915", "1917", "1896", "1897", 
                                      "1890", "1861", "1869", "1903", "1874", 
                                      "1865", "1854", "1918", "1915", "1916", 
                                      "1915", "1850", "1859", "1860", "1852", 
                                      "1859", "1860", "1860", "1855", "1854") 

docvars(ToxiCorpus, 'category') <- c("Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke", "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", "Historische Drucke", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher",
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Historische Drucke, Krieg 1914-1918", 
                                     "Krieg 1914-1918, Historische Drucke, Kinder- und Jugendbücher", 
                                     "Krieg 1914-1918, Historische Drucke, Kinder- und Jugendbücher", 
                                     "Krieg 1914-1918, Historische Drucke, Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher", "Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher, Historische Drucke", 
                                     "Kinder- und Jugendbücher", "Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher", "Kinder- und Jugendbücher", 
                                     "Kinder- und Jugendbücher") 

docvars(ToxiCorpus, 'publisher') <- c("Buchhandlung der Berliner Evangel. Missions-Gesellschaft", 
                                      "Walther", "ohne Verlag", "Aaba-Express", "Helmich", 
                                      "Hinrichs", "Mignon-Verlag", "Langen", "A. Weichert", 
                                      "C. Flemming", "Verl. des Tractathauses, Rodemeyer", 
                                      "Bagel", "Bagel", "Westermann", "Manz", "Lotzbeck", 
                                      "Schreiber & Schill", 
                                      "Verl. d. Nordd. Männer- u. Jünglingsbundes", 
                                      "Selbstverlag", "Union Dt. Verl.-Ges.", "Montanus", 
                                      "ohne Verlag", "Springer", "Oehmigke", "Auffarth", 
                                      "Spamer", "Spamer", "Spamer", "Spamer", "Steinkopf") 

docvars(ToxiCorpus, 'comment') <- c("PdruckeEuropeana1914-1918", "PdruckeAllg_bis1920", 
                                    "PdruckeEuropeana1914-1918", "PdruckeEuropeana1914-1918", 
                                    "PdruckeAllg_bis1920", "PdruckeEuropeana1914-1918", 
                                    "PdruckeEuropeana1914-1918", "PdruckeEuropeana1914-1918", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PdruckeEuropeana1914-1918", 
                                    "PdruckeEuropeana1914-1918", "PdruckeEuropeana1914-1918", 
                                    "PdruckeEuropeana1914-1918", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital", 
                                    "PSBBDrucke_WegehauptDigital", "PSBBDrucke_WegehauptDigital") 

